<?php

namespace App\Controller\Admin;

use App\Entity\Carousel;
use App\Form\CarouselType;
use App\Repository\CarouselRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/admin/carousel')]
final class CarouselController extends AbstractController
{
    #[Route(name: 'app_admin_carousel_index', methods: ['GET'])]
    public function index(CarouselRepository $carouselRepository): Response
    {
        return $this->render('admin/carousel/index.html.twig', [
            'carousels' => $carouselRepository->findBy([], [
                'rank' => 'ASC',
            ]),
            'maxRank' => $carouselRepository->findMaxRank(),
        ]);
    }

    #[Route('/new', name: 'app_admin_carousel_new', methods: ['GET', 'POST'])]
    public function new(Request $request, EntityManagerInterface $entityManager): Response
    {
        $carousel = new Carousel();
        $form = $this->createForm(CarouselType::class, $carousel);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->persist($carousel);
            $entityManager->flush();
            $this->addFlash('success', 'Le carousel a été ajouté avec succès');

            return $this->redirectToRoute('app_admin_carousel_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('admin/carousel/new.html.twig', [
            'carousel' => $carousel,
            'form' => $form,
        ]);
    }

    #[Route('/{id}/edit', name: 'app_admin_carousel_edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, Carousel $carousel, EntityManagerInterface $entityManager): Response
    {
        $form = $this->createForm(CarouselType::class, $carousel);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->flush();
            $this->addFlash('success', 'Le carousel a été modifié avec succès');

            return $this->redirectToRoute('app_admin_carousel_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('admin/carousel/edit.html.twig', [
            'carousel' => $carousel,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'app_admin_carousel_delete', methods: ['POST'])]
    public function delete(Request $request, Carousel $carousel, EntityManagerInterface $entityManager): Response
    {
        if ($this->isCsrfTokenValid('delete'.$carousel->getId(), $request->getPayload()->getString('_token'))) {
            $entityManager->remove($carousel);
            $entityManager->flush();
            $this->addFlash('success', 'Le carousel a été supprimé avec succès');
        } else {
            $this->addFlash('error', 'Le carousel n\'a pas pu être supprimé');
        }

        return $this->redirectToRoute('app_admin_carousel_index', [], Response::HTTP_SEE_OTHER);
    }

    #[Route('/up/{id}', name: 'app_admin_carousel_move_up', methods: ['POST'])]
    public function moveUp(Request $request, Carousel $carousel, EntityManagerInterface $em): Response
    {
        if ($this->isCsrfTokenValid('move-up'.$carousel->getId(), $request->getPayload()->getString('_token'))) {
            $carousel->setRank($carousel->getRank() + 1);
            $em->flush();
            $this->addFlash('success', 'Le rang du carousel a été avancé avec succès');
        } else {
            $this->addFlash('error', 'Le rang du carousel n\'a pas pu être avancé');
        }
        
        return $this->redirectToRoute('app_admin_carousel_index', [], Response::HTTP_SEE_OTHER);
    }
    
    #[Route('/down/{id}', name: 'app_admin_carousel_move_down', methods: ['POST'])]
    public function moveDown(Request $request, Carousel $carousel, EntityManagerInterface $em): Response
    {
        if ($carousel->getRank() > 0 && $this->isCsrfTokenValid('move-down'.$carousel->getId(), $request->getPayload()->getString('_token'))) {
            $carousel->setRank($carousel->getRank() - 1);
            $em->flush();
            $this->addFlash('success', 'Le rang du carousel a été reculé avec succès');
        } else {
            $this->addFlash('error', 'Le rang du carousel n\'a pas pu être reculé');
        }

        return $this->redirectToRoute('app_admin_carousel_index', [], Response::HTTP_SEE_OTHER);
    }
}
