<?php

namespace App\Controller\Admin;

use App\Entity\Menu;
use App\Form\MenuType;
use App\Repository\MenuRepository;
use App\Repository\SubmenuRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;

#[Route('/admin/menu')]
final class MenuController extends AbstractController
{
    #[Route(name: 'app_admin_menu_index', methods: ['GET'])]
    public function index(MenuRepository $menuRepository): Response
    {
        return $this->render('admin/menu/index.html.twig', [
            'menus' => $menuRepository->findBy([], [
                'rank' => 'ASC',
            ]),
            'maxRank' => $menuRepository->findMaxRank(),
        ]);
    }

    #[Route('/new', name: 'app_admin_menu_new', methods: ['GET', 'POST'])]
    public function new(Request $request, EntityManagerInterface $entityManager): Response
    {
        $menu = new Menu();
        $form = $this->createForm(MenuType::class, $menu);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->persist($menu);
            $entityManager->flush();
            $this->addFlash('success', 'Le menu a été ajouté avec succès');

            return $this->redirectToRoute('app_admin_menu_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('admin/menu/new.html.twig', [
            'menu' => $menu,
            'form' => $form,
        ]);
    }

    #[Route('/{id}/edit', name: 'app_admin_menu_edit', methods: ['GET', 'POST'])]
    public function edit(Request $request, Menu $menu, EntityManagerInterface $entityManager): Response
    {
        $form = $this->createForm(MenuType::class, $menu);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager->flush();
            $this->addFlash('success', 'Le menu a été modifié avec succès');

            return $this->redirectToRoute('app_admin_menu_index', [], Response::HTTP_SEE_OTHER);
        }

        return $this->render('admin/menu/edit.html.twig', [
            'menu' => $menu,
            'form' => $form,
        ]);
    }

    #[Route('/{id}', name: 'app_admin_menu_delete', methods: ['POST'])]
    public function delete(Request $request, Menu $menu, EntityManagerInterface $entityManager): Response
    {
        if ($this->isCsrfTokenValid('delete'.$menu->getId(), $request->getPayload()->getString('_token'))) {
            $entityManager->remove($menu);
            $entityManager->flush();
            $this->addFlash('success', 'Le menu a été supprimé avec succès');
        } else {
            $this->addFlash('success', 'Le menu n\'a pas pu être supprimé');
        }

        return $this->redirectToRoute('app_admin_menu_index', [], Response::HTTP_SEE_OTHER);
    }

    #[Route('/get-submenus/{menuId}', name: 'get_submenus')]
    public function getSubmenus(int $menuId, SubmenuRepository $subMenuRepository): JsonResponse
    {
        $submenus = $subMenuRepository->findBy(['menu' => $menuId]);

        $submenuData = array_map(function ($submenu) {
            return [
                'id' => $submenu->getId(),
                'title' => $submenu->getTitle(),
            ];
        }, $submenus);

        return new JsonResponse(['submenus' => $submenuData]);
    }

    #[Route('/up/{id}', name: 'app_admin_menu_move_up', methods: ['POST'])]
    public function moveUp(Request $request, Menu $menu, EntityManagerInterface $em): Response
    {
        if ($this->isCsrfTokenValid('move-up'.$menu->getId(), $request->getPayload()->getString('_token'))) {
            $menu->setRank($menu->getRank() + 1);
            $em->flush();
            $this->addFlash('success', 'Le rang du menu a été avancé avec succès');
        } else {
            $this->addFlash('error', 'Le rang du menu n\'a pas pu être avancé');
        }
        
        return $this->redirectToRoute('app_admin_menu_index', [], Response::HTTP_SEE_OTHER);
    }
    
    #[Route('/down/{id}', name: 'app_admin_menu_move_down', methods: ['POST'])]
    public function moveDown(Request $request, Menu $menu, EntityManagerInterface $em): Response
    {
        if ($menu->getRank() > 0 && $this->isCsrfTokenValid('move-down'.$menu->getId(), $request->getPayload()->getString('_token'))) {
            $menu->setRank($menu->getRank() - 1);
            $em->flush();
            $this->addFlash('success', 'Le rang du menu a été reculé avec succès');
        } else {
            $this->addFlash('error', 'Le rang du menu n\'a pas pu être reculé');
        }

        return $this->redirectToRoute('app_admin_menu_index', [], Response::HTTP_SEE_OTHER);
    }
}
