<?php

namespace App\Form;

use App\Entity\Menu;
use App\Entity\Page;
use App\Entity\Submenu;
use App\Repository\SubmenuRepository;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\Form\AbstractType;
use FOS\CKEditorBundle\Form\Type\CKEditorType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use CKSource\Bundle\CKFinderBundle\Form\Type\CKFinderFileChooserType;

class PageType extends AbstractType
{
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add('menu', EntityType::class, [
                'class' => Menu::class,
                'choice_label' => 'title',
                'placeholder' => 'Choisir un menu',
                ])
            ->add('submenu', EntityType::class, [
                'class' => Submenu::class,
                'choice_label' => 'title',
                'placeholder' => 'Choisir un sous-menu',
                'query_builder' => function (SubmenuRepository $repo) {
                        return $repo->createQueryBuilder('sm')
                            ->where('1 = 0');
                    },
            ])
            ->add('title', TextType::class, [
                'label' => 'Titre',
                'attr' => ['class' => 'form-control']
            ])
            ->add('description', CKEditorType::class, [
                'required' => false,
                'attr' => [
                    'rows' => 5
                ],
                'constraints' => [
                    new Assert\NotNull(),
                ],
            ])
            ->add('secondSection', CKEditorType::class, [
                'required' => false,
                'attr' => [
                    'rows' => 5
                ],
            ])
            ->add('isEnabled',  ChoiceType::class, [
                'choices' => [
                    'Oui' => true,
                    'Non' => false,
                ],
                'expanded' => true,
                'multiple' => false,
                'label' => 'Publié',
            ])
            ->add('image', CKFinderFileChooserType::class, [
                'label' => 'Image',
                'button_text' => 'Browse photos',
                'button_attr' => [
                    'class' => 'btn btn-secondary'
                ],
                'attr' => [
                    'class' => 'form-control', // Classe pour styliser le champ texte
                ],
                'row_attr' => [
                    'class' => 'input-group', // Aligne le champ texte et le bouton
                ],
            ])

        ;

        $builder->addEventListener(FormEvents::PRE_SET_DATA, function (FormEvent $event) {
            $form = $event->getForm();
            $page = $event->getData();

            if (null !== $page && null !== $page->getSubmenu()) {
                $menu = $page->getSubmenu()->getMenu();

                $form->add('submenu', EntityType::class, [
                    'class' => Submenu::class,
                    'choice_label' => 'title',
                    'placeholder' => 'Choisir un sous-menu',
                    'query_builder' => function (SubmenuRepository $repo) use ($menu) {
                        return $repo->createQueryBuilder('sm')
                            ->where('sm.menu = :menu')
                            ->setParameter('menu', $menu);
                    },
                ]);
            }
        });

        $builder->get('menu')->addEventListener(
            FormEvents::POST_SUBMIT,
            function (FormEvent $event) {
                $form = $event->getForm();
                $menu = $event->getForm()->getData();

                $form->getParent()->add('submenu', EntityType::class, [
                    'class' => Submenu::class,
                    'choice_label' => 'title',
                    'placeholder' => 'Choisir un sous-menu',
                    'query_builder' => function (SubmenuRepository $repo) use ($menu) {
                        return $repo->createQueryBuilder('sm')
                            ->where('sm.menu = :menu')
                            ->setParameter('menu', $menu);
                    },
                ]);
            }
        );
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => Page::class,
        ]);
    }
}
